// This example checks if a UDP server is up.
//
// The OPTION method is used to discover the server capabilities:
// - https://datatracker.ietf.org/doc/html/rfc3261#section-11
package main

import (
	sipgo "github.com/emiago/sipgo/sip"
	"github.com/vulncheck-oss/go-exploit/output"
	"github.com/vulncheck-oss/go-exploit/protocol"
	"github.com/vulncheck-oss/go-exploit/protocol/sip"
)

const (
	host = "127.0.0.1"
	// Most of the times the servers answer without them.
	fromUser = "bob"
	toUser   = "alice"
)

func main() {
	port := sip.DefaultPorts[sip.UDP]
	conn, ok := protocol.UDPConnect(host, port)
	if !ok {
		output.PrintfFrameworkError("Connecting to server %s:%d", host, port)

		return
	}
	defer conn.Close()
	reqOpts := sip.NewSipRequestOpts{
		Port:   port,
		ToUser: toUser,
		User:   fromUser,
	}
	req, ok := sip.NewSipRequest(sipgo.OPTIONS, host, &reqOpts)
	if !ok {
		output.PrintfFrameworkError("Creating request to %s with options %+v", host, reqOpts)

		return
	}
	resp, ok := sip.SendAndReceiveUDP(conn, req)
	if ok {
		output.PrintfFrameworkSuccess("Server is up, response: %s", resp.String())
	} else {
		output.PrintfFrameworkError("Sending request %s", req.String())
	}
}
