// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package workspaceevents provides access to the Google Workspace Events API.
//
// For product documentation, see: https://developers.google.com/workspace/events
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/workspaceevents/v1"
//	...
//	ctx := context.Background()
//	workspaceeventsService, err := workspaceevents.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// By default, all available scopes (see "Constants") are used to authenticate.
// To restrict scopes, use [google.golang.org/api/option.WithScopes]:
//
//	workspaceeventsService, err := workspaceevents.NewService(ctx, option.WithScopes(workspaceevents.MeetingsSpaceReadonlyScope))
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	workspaceeventsService, err := workspaceevents.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	workspaceeventsService, err := workspaceevents.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package workspaceevents // import "google.golang.org/api/workspaceevents/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "workspaceevents:v1"
const apiName = "workspaceevents"
const apiVersion = "v1"
const basePath = "https://workspaceevents.googleapis.com/"
const basePathTemplate = "https://workspaceevents.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://workspaceevents.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// On their own behalf, apps in Google Chat can see, add, update, and remove
	// members from conversations and spaces
	ChatAppMembershipsScope = "https://www.googleapis.com/auth/chat.app.memberships"

	// On their own behalf, apps in Google Chat can see all messages and their
	// associated reactions and message content
	ChatAppMessagesReadonlyScope = "https://www.googleapis.com/auth/chat.app.messages.readonly"

	// On their own behalf, apps in Google Chat can create conversations and spaces
	// and see or update their metadata (including history settings and access
	// settings)
	ChatAppSpacesScope = "https://www.googleapis.com/auth/chat.app.spaces"

	// Private Service: https://www.googleapis.com/auth/chat.bot
	ChatBotScope = "https://www.googleapis.com/auth/chat.bot"

	// See, add, update, and remove members from conversations and spaces in Google
	// Chat
	ChatMembershipsScope = "https://www.googleapis.com/auth/chat.memberships"

	// View members in Google Chat conversations.
	ChatMembershipsReadonlyScope = "https://www.googleapis.com/auth/chat.memberships.readonly"

	// See, compose, send, update, and delete messages as well as their message
	// content; add, see, and delete reactions to messages.
	ChatMessagesScope = "https://www.googleapis.com/auth/chat.messages"

	// See, add, and delete reactions as well as their reaction content to messages
	// in Google Chat
	ChatMessagesReactionsScope = "https://www.googleapis.com/auth/chat.messages.reactions"

	// View reactions as well as their reaction content to messages in Google Chat
	ChatMessagesReactionsReadonlyScope = "https://www.googleapis.com/auth/chat.messages.reactions.readonly"

	// See messages as well as their reactions and message content in Google Chat
	ChatMessagesReadonlyScope = "https://www.googleapis.com/auth/chat.messages.readonly"

	// Create conversations and spaces and see or update metadata (including
	// history settings and access settings) in Google Chat
	ChatSpacesScope = "https://www.googleapis.com/auth/chat.spaces"

	// View chat and spaces in Google Chat
	ChatSpacesReadonlyScope = "https://www.googleapis.com/auth/chat.spaces.readonly"

	// See, edit, create, and delete all of your Google Drive files
	DriveScope = "https://www.googleapis.com/auth/drive"

	// See, edit, create, and delete only the specific Google Drive files you use
	// with this app
	DriveFileScope = "https://www.googleapis.com/auth/drive.file"

	// View and manage metadata of files in your Google Drive
	DriveMetadataScope = "https://www.googleapis.com/auth/drive.metadata"

	// See information about your Google Drive files
	DriveMetadataReadonlyScope = "https://www.googleapis.com/auth/drive.metadata.readonly"

	// See and download all your Google Drive files
	DriveReadonlyScope = "https://www.googleapis.com/auth/drive.readonly"

	// Create, edit, and see information about your Google Meet conferences created
	// by the app.
	MeetingsSpaceCreatedScope = "https://www.googleapis.com/auth/meetings.space.created"

	// Read information about any of your Google Meet conferences
	MeetingsSpaceReadonlyScope = "https://www.googleapis.com/auth/meetings.space.readonly"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/chat.app.memberships",
		"https://www.googleapis.com/auth/chat.app.messages.readonly",
		"https://www.googleapis.com/auth/chat.app.spaces",
		"https://www.googleapis.com/auth/chat.bot",
		"https://www.googleapis.com/auth/chat.memberships",
		"https://www.googleapis.com/auth/chat.memberships.readonly",
		"https://www.googleapis.com/auth/chat.messages",
		"https://www.googleapis.com/auth/chat.messages.reactions",
		"https://www.googleapis.com/auth/chat.messages.reactions.readonly",
		"https://www.googleapis.com/auth/chat.messages.readonly",
		"https://www.googleapis.com/auth/chat.spaces",
		"https://www.googleapis.com/auth/chat.spaces.readonly",
		"https://www.googleapis.com/auth/drive",
		"https://www.googleapis.com/auth/drive.file",
		"https://www.googleapis.com/auth/drive.metadata",
		"https://www.googleapis.com/auth/drive.metadata.readonly",
		"https://www.googleapis.com/auth/drive.readonly",
		"https://www.googleapis.com/auth/meetings.space.created",
		"https://www.googleapis.com/auth/meetings.space.readonly",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.Message = NewMessageService(s)
	s.Operations = NewOperationsService(s)
	s.Subscriptions = NewSubscriptionsService(s)
	s.Tasks = NewTasksService(s)
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Message *MessageService

	Operations *OperationsService

	Subscriptions *SubscriptionsService

	Tasks *TasksService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewMessageService(s *Service) *MessageService {
	rs := &MessageService{s: s}
	return rs
}

type MessageService struct {
	s *Service
}

func NewOperationsService(s *Service) *OperationsService {
	rs := &OperationsService{s: s}
	return rs
}

type OperationsService struct {
	s *Service
}

func NewSubscriptionsService(s *Service) *SubscriptionsService {
	rs := &SubscriptionsService{s: s}
	return rs
}

type SubscriptionsService struct {
	s *Service
}

func NewTasksService(s *Service) *TasksService {
	rs := &TasksService{s: s}
	rs.PushNotificationConfigs = NewTasksPushNotificationConfigsService(s)
	return rs
}

type TasksService struct {
	s *Service

	PushNotificationConfigs *TasksPushNotificationConfigsService
}

func NewTasksPushNotificationConfigsService(s *Service) *TasksPushNotificationConfigsService {
	rs := &TasksPushNotificationConfigsService{s: s}
	return rs
}

type TasksPushNotificationConfigsService struct {
	s *Service
}

// Artifact: Artifacts are the container for task completed results. These are
// similar to Messages but are intended to be the product of a task, as opposed
// to point-to-point communication.
type Artifact struct {
	// ArtifactId: Unique identifier (e.g. UUID) for the artifact. It must be at
	// least unique within a task.
	ArtifactId string `json:"artifactId,omitempty"`
	// Description: A human readable description of the artifact, optional.
	Description string `json:"description,omitempty"`
	// Extensions: The URIs of extensions that are present or contributed to this
	// Artifact.
	Extensions []string `json:"extensions,omitempty"`
	// Metadata: Optional metadata included with the artifact.
	Metadata googleapi.RawMessage `json:"metadata,omitempty"`
	// Name: A human readable name for the artifact.
	Name string `json:"name,omitempty"`
	// Parts: The content of the artifact.
	Parts []*Part `json:"parts,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ArtifactId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ArtifactId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Artifact) MarshalJSON() ([]byte, error) {
	type NoMethod Artifact
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AuthenticationInfo: Defines authentication details, used for push
// notifications.
type AuthenticationInfo struct {
	// Credentials: Optional credentials
	Credentials string `json:"credentials,omitempty"`
	// Schemes: Supported authentication schemes - e.g. Basic, Bearer, etc
	Schemes []string `json:"schemes,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Credentials") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Credentials") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AuthenticationInfo) MarshalJSON() ([]byte, error) {
	type NoMethod AuthenticationInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type CancelTaskRequest struct {
}

// DataPart: DataPart represents a structured blob. This is most commonly a
// JSON payload.
type DataPart struct {
	Data googleapi.RawMessage `json:"data,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Data") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Data") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DataPart) MarshalJSON() ([]byte, error) {
	type NoMethod DataPart
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use it as
// the request or the response type of an API method. For instance: service Foo
// { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// FilePart: FilePart represents the different ways files can be provided. If
// files are small, directly feeding the bytes is supported via
// file_with_bytes. If the file is large, the agent should read the content as
// appropriate directly from the file_with_uri source.
type FilePart struct {
	FileWithBytes string `json:"fileWithBytes,omitempty"`
	FileWithUri   string `json:"fileWithUri,omitempty"`
	MimeType      string `json:"mimeType,omitempty"`
	Name          string `json:"name,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FileWithBytes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FileWithBytes") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FilePart) MarshalJSON() ([]byte, error) {
	type NoMethod FilePart
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListSubscriptionsResponse: The response message for
// SubscriptionsService.ListSubscriptions.
type ListSubscriptionsResponse struct {
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Subscriptions: List of subscriptions.
	Subscriptions []*Subscription `json:"subscriptions,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListSubscriptionsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListSubscriptionsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type ListTaskPushNotificationConfigResponse struct {
	// Configs: The list of push notification configurations.
	Configs []*TaskPushNotificationConfig `json:"configs,omitempty"`
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Configs") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Configs") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListTaskPushNotificationConfigResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListTaskPushNotificationConfigResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Message: Message is one unit of communication between client and server. It
// is associated with a context and optionally a task. Since the server is
// responsible for the context definition, it must always provide a context_id
// in its messages. The client can optionally provide the context_id if it
// knows the context to associate the message to. Similarly for task_id, except
// the server decides if a task is created and whether to include the task_id.
type Message struct {
	// Content: protolint:disable REPEATED_FIELD_NAMES_PLURALIZED Content is the
	// container of the message content.
	Content []*Part `json:"content,omitempty"`
	// ContextId: The context id of the message. This is optional and if set, the
	// message will be associated with the given context.
	ContextId string `json:"contextId,omitempty"`
	// Extensions: The URIs of extensions that are present or contributed to this
	// Message.
	Extensions []string `json:"extensions,omitempty"`
	// MessageId: The unique identifier (e.g. UUID)of the message. This is required
	// and created by the message creator.
	MessageId string `json:"messageId,omitempty"`
	// Metadata: protolint:enable REPEATED_FIELD_NAMES_PLURALIZED Any optional
	// metadata to provide along with the message.
	Metadata googleapi.RawMessage `json:"metadata,omitempty"`
	// Role: A role for the message.
	//
	// Possible values:
	//   "ROLE_UNSPECIFIED"
	//   "ROLE_USER" - USER role refers to communication from the client to the
	// server.
	//   "ROLE_AGENT" - AGENT role refers to communication from the server to the
	// client.
	Role string `json:"role,omitempty"`
	// TaskId: The task id of the message. This is optional and if set, the message
	// will be associated with the given task.
	TaskId string `json:"taskId,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Content") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Content") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Message) MarshalJSON() ([]byte, error) {
	type NoMethod Message
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// NotificationEndpoint: The endpoint where the subscription delivers events.
type NotificationEndpoint struct {
	// PubsubTopic: Immutable. The Pub/Sub topic that receives events for the
	// subscription. Format: `projects/{project}/topics/{topic}` You must create
	// the topic in the same Google Cloud project where you create this
	// subscription. Note: The Google Workspace Events API uses ordering keys
	// (https://cloud.google.com/pubsub/docs/ordering) for the benefit of
	// sequential events. If the Cloud Pub/Sub topic has a message storage policy
	// (https://cloud.google.com/pubsub/docs/resource-location-restriction#exceptions)
	// configured to exclude the nearest Google Cloud region, publishing events
	// with ordering keys will fail. When the topic receives events, the events are
	// encoded as Pub/Sub messages. For details, see the Google Cloud Pub/Sub
	// Protocol Binding for CloudEvents
	// (https://github.com/googleapis/google-cloudevents/blob/main/docs/spec/pubsub.md).
	PubsubTopic string `json:"pubsubTopic,omitempty"`
	// ForceSendFields is a list of field names (e.g. "PubsubTopic") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "PubsubTopic") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s NotificationEndpoint) MarshalJSON() ([]byte, error) {
	type NoMethod NotificationEndpoint
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Operation: This resource represents a long-running operation that is the
// result of a network API call.
type Operation struct {
	// Done: If the value is `false`, it means the operation is still in progress.
	// If `true`, the operation is completed, and either `error` or `response` is
	// available.
	Done bool `json:"done,omitempty"`
	// Error: The error result of the operation in case of failure or cancellation.
	Error *Status `json:"error,omitempty"`
	// Metadata: Service-specific metadata associated with the operation. It
	// typically contains progress information and common metadata such as create
	// time. Some services might not provide such metadata. Any method that returns
	// a long-running operation should document the metadata type, if any.
	Metadata googleapi.RawMessage `json:"metadata,omitempty"`
	// Name: The server-assigned name, which is only unique within the same service
	// that originally returns it. If you use the default HTTP mapping, the `name`
	// should be a resource name ending with `operations/{unique_id}`.
	Name string `json:"name,omitempty"`
	// Response: The normal, successful response of the operation. If the original
	// method returns no data on success, such as `Delete`, the response is
	// `google.protobuf.Empty`. If the original method is standard
	// `Get`/`Create`/`Update`, the response should be the resource. For other
	// methods, the response should have the type `XxxResponse`, where `Xxx` is the
	// original method name. For example, if the original method name is
	// `TakeSnapshot()`, the inferred response type is `TakeSnapshotResponse`.
	Response googleapi.RawMessage `json:"response,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Done") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Done") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Operation) MarshalJSON() ([]byte, error) {
	type NoMethod Operation
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Part: Part represents a container for a section of communication content.
// Parts can be purely textual, some sort of file (image, video, etc) or a
// structured data blob (i.e. JSON).
type Part struct {
	Data *DataPart `json:"data,omitempty"`
	File *FilePart `json:"file,omitempty"`
	// Metadata: Optional metadata associated with this part.
	Metadata googleapi.RawMessage `json:"metadata,omitempty"`
	Text     string               `json:"text,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Data") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Data") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Part) MarshalJSON() ([]byte, error) {
	type NoMethod Part
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PayloadOptions: Options about what data to include in the event payload.
// Only supported for Google Chat and Google Drive events.
type PayloadOptions struct {
	// FieldMask: Optional. If `include_resource` is set to `true`, the list of
	// fields to include in the event payload. Separate fields with a comma. For
	// example, to include a Google Chat message's sender and create time, enter
	// `message.sender,message.createTime`. If omitted, the payload includes all
	// fields for the resource. If you specify a field that doesn't exist for the
	// resource, the system ignores the field.
	FieldMask string `json:"fieldMask,omitempty"`
	// IncludeResource: Optional. Whether the event payload includes data about the
	// resource that changed. For example, for an event where a Google Chat message
	// was created, whether the payload contains data about the `Message`
	// (https://developers.google.com/chat/api/reference/rest/v1/spaces.messages)
	// resource. If false, the event payload only includes the name of the changed
	// resource.
	IncludeResource bool `json:"includeResource,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FieldMask") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FieldMask") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PayloadOptions) MarshalJSON() ([]byte, error) {
	type NoMethod PayloadOptions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PushNotificationConfig: Configuration for setting up push notifications for
// task updates.
type PushNotificationConfig struct {
	// Authentication: Information about the authentication to sent with the
	// notification
	Authentication *AuthenticationInfo `json:"authentication,omitempty"`
	// Id: A unique identifier (e.g. UUID) for this push notification.
	Id string `json:"id,omitempty"`
	// Token: Token unique for this task/session
	Token string `json:"token,omitempty"`
	// Url: Url to send the notification too
	Url string `json:"url,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Authentication") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Authentication") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PushNotificationConfig) MarshalJSON() ([]byte, error) {
	type NoMethod PushNotificationConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReactivateSubscriptionRequest: The request message for
// SubscriptionsService.ReactivateSubscription.
type ReactivateSubscriptionRequest struct {
}

// SendMessageConfiguration: Configuration of a send message request.
type SendMessageConfiguration struct {
	// AcceptedOutputModes: The output modes that the agent is expected to respond
	// with.
	AcceptedOutputModes []string `json:"acceptedOutputModes,omitempty"`
	// Blocking: If true, the message will be blocking until the task is completed.
	// If false, the message will be non-blocking and the task will be returned
	// immediately. It is the caller's responsibility to check for any task
	// updates.
	Blocking bool `json:"blocking,omitempty"`
	// HistoryLength: The maximum number of messages to include in the history. if
	// 0, the history will be unlimited.
	HistoryLength int64 `json:"historyLength,omitempty"`
	// PushNotification: A configuration of a webhook that can be used to receive
	// updates
	PushNotification *PushNotificationConfig `json:"pushNotification,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AcceptedOutputModes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AcceptedOutputModes") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SendMessageConfiguration) MarshalJSON() ([]byte, error) {
	type NoMethod SendMessageConfiguration
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SendMessageRequest: /////////// Request Messages ///////////
type SendMessageRequest struct {
	// Configuration: Configuration for the send request.
	Configuration *SendMessageConfiguration `json:"configuration,omitempty"`
	// Message: Required. The message to send to the agent.
	Message *Message `json:"message,omitempty"`
	// Metadata: Optional metadata for the request.
	Metadata googleapi.RawMessage `json:"metadata,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Configuration") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Configuration") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SendMessageRequest) MarshalJSON() ([]byte, error) {
	type NoMethod SendMessageRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Status: The `Status` type defines a logical error model that is suitable for
// different programming environments, including REST APIs and RPC APIs. It is
// used by gRPC (https://github.com/grpc). Each `Status` message contains three
// pieces of data: error code, error message, and error details. You can find
// out more about this error model and how to work with it in the API Design
// Guide (https://cloud.google.com/apis/design/errors).
type Status struct {
	// Code: The status code, which should be an enum value of google.rpc.Code.
	Code int64 `json:"code,omitempty"`
	// Details: A list of messages that carry the error details. There is a common
	// set of message types for APIs to use.
	Details []googleapi.RawMessage `json:"details,omitempty"`
	// Message: A developer-facing error message, which should be in English. Any
	// user-facing error message should be localized and sent in the
	// google.rpc.Status.details field, or localized by the client.
	Message string `json:"message,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Code") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Code") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Status) MarshalJSON() ([]byte, error) {
	type NoMethod Status
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// StreamResponse: The stream response for a message. The stream should be one
// of the following sequences: If the response is a message, the stream should
// contain one, and only one, message and then close If the response is a task
// lifecycle, the first response should be a Task object followed by zero or
// more TaskStatusUpdateEvents and TaskArtifactUpdateEvents. The stream should
// complete when the Task if in an interrupted or terminal state. A stream that
// ends before these conditions are met are
type StreamResponse struct {
	ArtifactUpdate *TaskArtifactUpdateEvent `json:"artifactUpdate,omitempty"`
	Message        *Message                 `json:"message,omitempty"`
	StatusUpdate   *TaskStatusUpdateEvent   `json:"statusUpdate,omitempty"`
	Task           *Task                    `json:"task,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ArtifactUpdate") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ArtifactUpdate") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s StreamResponse) MarshalJSON() ([]byte, error) {
	type NoMethod StreamResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Subscription: A subscription to receive events about a Google Workspace
// resource. To learn more about subscriptions, see the Google Workspace Events
// API overview (https://developers.google.com/workspace/events).
type Subscription struct {
	// Authority: Output only. The user who authorized the creation of the
	// subscription. When a user authorizes the subscription, this field and the
	// `user_authority` field have the same value and the format is: Format:
	// `users/{user}` For Google Workspace users, the `{user}` value is the
	// `user.id`
	// (https://developers.google.com/admin-sdk/directory/reference/rest/v1/users#User.FIELDS.ids)
	// field from the Directory API. When a Chat app authorizes the subscription,
	// only `service_account_authority` field populates and this field is empty.
	Authority string `json:"authority,omitempty"`
	// CreateTime: Output only. The time when the subscription is created.
	CreateTime string `json:"createTime,omitempty"`
	// Etag: Optional. This checksum is computed by the server based on the value
	// of other fields, and might be sent on update requests to ensure the client
	// has an up-to-date value before proceeding.
	Etag string `json:"etag,omitempty"`
	// EventTypes: Required. Unordered list. Input for creating a subscription.
	// Otherwise, output only. One or more types of events to receive about the
	// target resource. Formatted according to the CloudEvents specification. The
	// supported event types depend on the target resource of your subscription.
	// For details, see Supported Google Workspace events
	// (https://developers.google.com/workspace/events/guides#supported-events). By
	// default, you also receive events about the lifecycle of your subscription
	// (https://developers.google.com/workspace/events/guides/events-lifecycle).
	// You don't need to specify lifecycle events for this field. If you specify an
	// event type that doesn't exist for the target resource, the request returns
	// an HTTP `400 Bad Request` status code.
	EventTypes []string `json:"eventTypes,omitempty"`
	// ExpireTime: Non-empty default. The timestamp in UTC when the subscription
	// expires. Always displayed on output, regardless of what was used on input.
	ExpireTime string `json:"expireTime,omitempty"`
	// Name: Identifier. Resource name of the subscription. Format:
	// `subscriptions/{subscription}`
	Name string `json:"name,omitempty"`
	// NotificationEndpoint: Required. Immutable. The endpoint where the
	// subscription delivers events, such as a Pub/Sub topic.
	NotificationEndpoint *NotificationEndpoint `json:"notificationEndpoint,omitempty"`
	// PayloadOptions: Optional. Options about what data to include in the event
	// payload. Only supported for Google Chat and Google Drive events.
	PayloadOptions *PayloadOptions `json:"payloadOptions,omitempty"`
	// Reconciling: Output only. If `true`, the subscription is in the process of
	// being updated.
	Reconciling bool `json:"reconciling,omitempty"`
	// State: Output only. The state of the subscription. Determines whether the
	// subscription can receive events and deliver them to the notification
	// endpoint.
	//
	// Possible values:
	//   "STATE_UNSPECIFIED" - Default value. This value is unused.
	//   "ACTIVE" - The subscription is active and can receive and deliver events
	// to its notification endpoint.
	//   "SUSPENDED" - The subscription is unable to receive events due to an
	// error. To identify the error, see the `suspension_reason` field.
	//   "DELETED" - The subscription is deleted.
	State string `json:"state,omitempty"`
	// SuspensionReason: Output only. The error that suspended the subscription. To
	// reactivate the subscription, resolve the error and call the
	// `ReactivateSubscription` method.
	//
	// Possible values:
	//   "ERROR_TYPE_UNSPECIFIED" - Default value. This value is unused.
	//   "USER_SCOPE_REVOKED" - The authorizing user has revoked the grant of one
	// or more OAuth scopes. To learn more about authorization for Google
	// Workspace, see [Configure the OAuth consent
	// screen](https://developers.google.com/workspace/guides/configure-oauth-consen
	// t#choose-scopes).
	//   "RESOURCE_DELETED" - The target resource for the subscription no longer
	// exists.
	//   "USER_AUTHORIZATION_FAILURE" - The user that authorized the creation of
	// the subscription no longer has access to the subscription's target resource.
	//   "ENDPOINT_PERMISSION_DENIED" - The Google Workspace application doesn't
	// have access to deliver events to your subscription's notification endpoint.
	//   "ENDPOINT_NOT_FOUND" - The subscription's notification endpoint doesn't
	// exist, or the endpoint can't be found in the Google Cloud project where you
	// created the subscription.
	//   "ENDPOINT_RESOURCE_EXHAUSTED" - The subscription's notification endpoint
	// failed to receive events due to insufficient quota or reaching rate
	// limiting.
	//   "OTHER" - An unidentified error has occurred.
	SuspensionReason string `json:"suspensionReason,omitempty"`
	// TargetResource: Required. Immutable. The Google Workspace resource that's
	// monitored for events, formatted as the full resource name
	// (https://google.aip.dev/122#full-resource-names). To learn about target
	// resources and the events that they support, see Supported Google Workspace
	// events (https://developers.google.com/workspace/events#supported-events). A
	// user can only authorize your app to create one subscription for a given
	// target resource. If your app tries to create another subscription with the
	// same user credentials, the request returns an `ALREADY_EXISTS` error.
	TargetResource string `json:"targetResource,omitempty"`
	// Ttl: Input only. The time-to-live (TTL) or duration for the subscription. If
	// unspecified or set to `0`, uses the maximum possible duration.
	Ttl string `json:"ttl,omitempty"`
	// Uid: Output only. System-assigned unique identifier for the subscription.
	Uid string `json:"uid,omitempty"`
	// UpdateTime: Output only. The last time that the subscription is updated.
	UpdateTime string `json:"updateTime,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Authority") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Authority") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Subscription) MarshalJSON() ([]byte, error) {
	type NoMethod Subscription
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Task: Task is the core unit of action for A2A. It has a current status and
// when results are created for the task they are stored in the artifact. If
// there are multiple turns for a task, these are stored in history.
type Task struct {
	// Artifacts: A set of output artifacts for a Task.
	Artifacts []*Artifact `json:"artifacts,omitempty"`
	// ContextId: Unique identifier (e.g. UUID) for the contextual collection of
	// interactions (tasks and messages). Created by the A2A server.
	ContextId string `json:"contextId,omitempty"`
	// History: protolint:disable REPEATED_FIELD_NAMES_PLURALIZED The history of
	// interactions from a task.
	History []*Message `json:"history,omitempty"`
	// Id: Unique identifier (e.g. UUID) for the task, generated by the server for
	// a new task.
	Id string `json:"id,omitempty"`
	// Metadata: protolint:enable REPEATED_FIELD_NAMES_PLURALIZED A key/value
	// object to store custom metadata about a task.
	Metadata googleapi.RawMessage `json:"metadata,omitempty"`
	// Status: The current status of a Task, including state and a message.
	Status *TaskStatus `json:"status,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Artifacts") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Artifacts") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Task) MarshalJSON() ([]byte, error) {
	type NoMethod Task
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TaskArtifactUpdateEvent: TaskArtifactUpdateEvent represents a task delta
// where an artifact has been generated.
type TaskArtifactUpdateEvent struct {
	// Append: Whether this should be appended to a prior one produced
	Append bool `json:"append,omitempty"`
	// Artifact: The artifact itself
	Artifact *Artifact `json:"artifact,omitempty"`
	// ContextId: The id of the context that this task belongs too
	ContextId string `json:"contextId,omitempty"`
	// LastChunk: Whether this represents the last part of an artifact
	LastChunk bool `json:"lastChunk,omitempty"`
	// Metadata: Optional metadata associated with the artifact update.
	Metadata googleapi.RawMessage `json:"metadata,omitempty"`
	// TaskId: The id of the task for this artifact
	TaskId string `json:"taskId,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Append") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Append") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TaskArtifactUpdateEvent) MarshalJSON() ([]byte, error) {
	type NoMethod TaskArtifactUpdateEvent
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type TaskPushNotificationConfig struct {
	// Name: The resource name of the config. Format:
	// tasks/{task_id}/pushNotificationConfigs/{config_id}
	Name string `json:"name,omitempty"`
	// PushNotificationConfig: The push notification configuration details.
	PushNotificationConfig *PushNotificationConfig `json:"pushNotificationConfig,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Name") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Name") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TaskPushNotificationConfig) MarshalJSON() ([]byte, error) {
	type NoMethod TaskPushNotificationConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TaskStatus: A container for the status of a task
type TaskStatus struct {
	// Message: A message associated with the status.
	Message *Message `json:"message,omitempty"`
	// State: The current state of this task
	//
	// Possible values:
	//   "TASK_STATE_UNSPECIFIED"
	//   "TASK_STATE_SUBMITTED" - Represents the status that acknowledges a task is
	// created
	//   "TASK_STATE_WORKING" - Represents the status that a task is actively being
	// processed
	//   "TASK_STATE_COMPLETED" - Represents the status a task is finished. This is
	// a terminal state
	//   "TASK_STATE_FAILED" - Represents the status a task is done but failed.
	// This is a terminal state
	//   "TASK_STATE_CANCELLED" - Represents the status a task was cancelled before
	// it finished. This is a terminal state.
	//   "TASK_STATE_INPUT_REQUIRED" - Represents the status that the task requires
	// information to complete. This is an interrupted state.
	//   "TASK_STATE_REJECTED" - Represents the status that the agent has decided
	// to not perform the task. This may be done during initial task creation or
	// later once an agent has determined it can't or won't proceed. This is a
	// terminal state.
	//   "TASK_STATE_AUTH_REQUIRED" - Represents the state that some authentication
	// is needed from the upstream client. Authentication is expected to come
	// out-of-band thus this is not an interrupted or terminal state.
	State string `json:"state,omitempty"`
	// Timestamp: Timestamp when the status was recorded. Example:
	// "2023-10-27T10:00:00Z"
	Timestamp string `json:"timestamp,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Message") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Message") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TaskStatus) MarshalJSON() ([]byte, error) {
	type NoMethod TaskStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TaskStatusUpdateEvent: TaskStatusUpdateEvent is a delta even on a task
// indicating that a task has changed.
type TaskStatusUpdateEvent struct {
	// ContextId: The id of the context that the task belongs to
	ContextId string `json:"contextId,omitempty"`
	// Final: Whether this is the last status update expected for this task.
	Final bool `json:"final,omitempty"`
	// Metadata: Optional metadata to associate with the task update.
	Metadata googleapi.RawMessage `json:"metadata,omitempty"`
	// Status: The new status of the task.
	Status *TaskStatus `json:"status,omitempty"`
	// TaskId: The id of the task that is changed
	TaskId string `json:"taskId,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ContextId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ContextId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TaskStatusUpdateEvent) MarshalJSON() ([]byte, error) {
	type NoMethod TaskStatusUpdateEvent
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type MessageStreamCall struct {
	s                  *Service
	sendmessagerequest *SendMessageRequest
	urlParams_         gensupport.URLParams
	ctx_               context.Context
	header_            http.Header
}

// Stream: SendStreamingMessage is a streaming call that will return a stream
// of task update events until the Task is in an interrupted or terminal state.
func (r *MessageService) Stream(sendmessagerequest *SendMessageRequest) *MessageStreamCall {
	c := &MessageStreamCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.sendmessagerequest = sendmessagerequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *MessageStreamCall) Fields(s ...googleapi.Field) *MessageStreamCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *MessageStreamCall) Context(ctx context.Context) *MessageStreamCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *MessageStreamCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *MessageStreamCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.sendmessagerequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/message:stream")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "workspaceevents.message.stream", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "workspaceevents.message.stream" call.
// Any non-2xx status code is an error. Response headers are in either
// *StreamResponse.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *MessageStreamCall) Do(opts ...googleapi.CallOption) (*StreamResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &StreamResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "workspaceevents.message.stream", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type OperationsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the latest state of a long-running operation. Clients can use this
// method to poll the operation result at intervals as recommended by the API
// service.
//
// - name: The name of the operation resource.
func (r *OperationsService) Get(name string) *OperationsGetCall {
	c := &OperationsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OperationsGetCall) Fields(s ...googleapi.Field) *OperationsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *OperationsGetCall) IfNoneMatch(entityTag string) *OperationsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OperationsGetCall) Context(ctx context.Context) *OperationsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OperationsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OperationsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "workspaceevents.operations.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "workspaceevents.operations.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *OperationsGetCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "workspaceevents.operations.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SubscriptionsCreateCall struct {
	s            *Service
	subscription *Subscription
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Create: Creates a Google Workspace subscription. To learn how to use this
// method, see Create a Google Workspace subscription
// (https://developers.google.com/workspace/events/guides/create-subscription).
// For a subscription on a Chat target resource
// (https://developers.google.com/workspace/events/guides/events-chat), you can
// create a subscription as: - A Chat app by specifying an authorization scope
// that begins with `chat.app` and getting one-time administrator approval
// (Developer Preview (https://developers.google.com/workspace/preview)). To
// learn more, see Authorize as a Chat app with administrator approval
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app).
// - A user by specifying an authorization scope that doesn't include `app` in
// its name. To learn more, see Authorize as a Chat user
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user).
func (r *SubscriptionsService) Create(subscription *Subscription) *SubscriptionsCreateCall {
	c := &SubscriptionsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.subscription = subscription
	return c
}

// ValidateOnly sets the optional parameter "validateOnly": If set to `true`,
// validates and previews the request, but doesn't create the subscription.
func (c *SubscriptionsCreateCall) ValidateOnly(validateOnly bool) *SubscriptionsCreateCall {
	c.urlParams_.Set("validateOnly", fmt.Sprint(validateOnly))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SubscriptionsCreateCall) Fields(s ...googleapi.Field) *SubscriptionsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SubscriptionsCreateCall) Context(ctx context.Context) *SubscriptionsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SubscriptionsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SubscriptionsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.subscription)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/subscriptions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "workspaceevents.subscriptions.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "workspaceevents.subscriptions.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SubscriptionsCreateCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "workspaceevents.subscriptions.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SubscriptionsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a Google Workspace subscription. To learn how to use this
// method, see Delete a Google Workspace subscription
// (https://developers.google.com/workspace/events/guides/delete-subscription).
//
//   - name: Resource name of the subscription to delete. Format:
//     `subscriptions/{subscription}`.
func (r *SubscriptionsService) Delete(name string) *SubscriptionsDeleteCall {
	c := &SubscriptionsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// AllowMissing sets the optional parameter "allowMissing": If set to `true`
// and the subscription isn't found, the request succeeds but doesn't delete
// the subscription.
func (c *SubscriptionsDeleteCall) AllowMissing(allowMissing bool) *SubscriptionsDeleteCall {
	c.urlParams_.Set("allowMissing", fmt.Sprint(allowMissing))
	return c
}

// Etag sets the optional parameter "etag": Etag of the subscription. If
// present, it must match with the server's etag. Otherwise, request fails with
// the status `ABORTED`.
func (c *SubscriptionsDeleteCall) Etag(etag string) *SubscriptionsDeleteCall {
	c.urlParams_.Set("etag", etag)
	return c
}

// ValidateOnly sets the optional parameter "validateOnly": If set to `true`,
// validates and previews the request, but doesn't delete the subscription.
func (c *SubscriptionsDeleteCall) ValidateOnly(validateOnly bool) *SubscriptionsDeleteCall {
	c.urlParams_.Set("validateOnly", fmt.Sprint(validateOnly))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SubscriptionsDeleteCall) Fields(s ...googleapi.Field) *SubscriptionsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SubscriptionsDeleteCall) Context(ctx context.Context) *SubscriptionsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SubscriptionsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SubscriptionsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "workspaceevents.subscriptions.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "workspaceevents.subscriptions.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SubscriptionsDeleteCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "workspaceevents.subscriptions.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SubscriptionsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets details about a Google Workspace subscription. To learn how to use
// this method, see Get details about a Google Workspace subscription
// (https://developers.google.com/workspace/events/guides/get-subscription).
//
//   - name: Resource name of the subscription. Format:
//     `subscriptions/{subscription}`.
func (r *SubscriptionsService) Get(name string) *SubscriptionsGetCall {
	c := &SubscriptionsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SubscriptionsGetCall) Fields(s ...googleapi.Field) *SubscriptionsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SubscriptionsGetCall) IfNoneMatch(entityTag string) *SubscriptionsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SubscriptionsGetCall) Context(ctx context.Context) *SubscriptionsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SubscriptionsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SubscriptionsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "workspaceevents.subscriptions.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "workspaceevents.subscriptions.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Subscription.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SubscriptionsGetCall) Do(opts ...googleapi.CallOption) (*Subscription, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Subscription{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "workspaceevents.subscriptions.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SubscriptionsListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists Google Workspace subscriptions. To learn how to use this method,
// see List Google Workspace subscriptions
// (https://developers.google.com/workspace/events/guides/list-subscriptions).
func (r *SubscriptionsService) List() *SubscriptionsListCall {
	c := &SubscriptionsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Filter sets the optional parameter "filter": Required. A query filter. You
// can filter subscriptions by event type (`event_types`) and target resource
// (`target_resource`). You must specify at least one event type in your query.
// To filter for multiple event types, use the `OR` operator. To filter by both
// event type and target resource, use the `AND` operator and specify the full
// resource name, such as `//chat.googleapis.com/spaces/{space}`. For example,
// the following queries are valid: ```
// event_types:"google.workspace.chat.membership.v1.updated" OR
// event_types:"google.workspace.chat.message.v1.created"
// event_types:"google.workspace.chat.message.v1.created" AND
// target_resource="//chat.googleapis.com/spaces/{space}" (
// event_types:"google.workspace.chat.membership.v1.updated" OR
// event_types:"google.workspace.chat.message.v1.created" ) AND
// target_resource="//chat.googleapis.com/spaces/{space}" ``` The server
// rejects invalid queries with an `INVALID_ARGUMENT` error.
func (c *SubscriptionsListCall) Filter(filter string) *SubscriptionsListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// subscriptions to return. The service might return fewer than this value. If
// unspecified or set to `0`, up to 50 subscriptions are returned. The maximum
// value is 100. If you specify a value more than 100, the system only returns
// 100 subscriptions.
func (c *SubscriptionsListCall) PageSize(pageSize int64) *SubscriptionsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous list subscriptions call. Provide this parameter to retrieve
// the subsequent page. When paginating, the filter value should match the call
// that provided the page token. Passing a different value might lead to
// unexpected results.
func (c *SubscriptionsListCall) PageToken(pageToken string) *SubscriptionsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SubscriptionsListCall) Fields(s ...googleapi.Field) *SubscriptionsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SubscriptionsListCall) IfNoneMatch(entityTag string) *SubscriptionsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SubscriptionsListCall) Context(ctx context.Context) *SubscriptionsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SubscriptionsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SubscriptionsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/subscriptions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "workspaceevents.subscriptions.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "workspaceevents.subscriptions.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListSubscriptionsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *SubscriptionsListCall) Do(opts ...googleapi.CallOption) (*ListSubscriptionsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListSubscriptionsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "workspaceevents.subscriptions.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *SubscriptionsListCall) Pages(ctx context.Context, f func(*ListSubscriptionsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type SubscriptionsPatchCall struct {
	s            *Service
	name         string
	subscription *Subscription
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Patch: Updates or renews a Google Workspace subscription. To learn how to
// use this method, see Update or renew a Google Workspace subscription
// (https://developers.google.com/workspace/events/guides/update-subscription).
// For a subscription on a Chat target resource
// (https://developers.google.com/workspace/events/guides/events-chat), you can
// update a subscription as: - A Chat app by specifying an authorization scope
// that begins with `chat.app` andgetting one-time administrator approval
// (Developer Preview (https://developers.google.com/workspace/preview)). To
// learn more, see Authorize as a Chat app with administrator approval
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app).
// - A user by specifying an authorization scope that doesn't include `app` in
// its name. To learn more, see Authorize as a Chat user
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user).
//
//   - name: Identifier. Resource name of the subscription. Format:
//     `subscriptions/{subscription}`.
func (r *SubscriptionsService) Patch(name string, subscription *Subscription) *SubscriptionsPatchCall {
	c := &SubscriptionsPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.subscription = subscription
	return c
}

// UpdateMask sets the optional parameter "updateMask": The field to update. If
// omitted, updates any fields included in the request. You can update one of
// the following fields in a subscription: * `expire_time`: The timestamp when
// the subscription expires. * `ttl`: The time-to-live (TTL) or duration of the
// subscription. * `event_types`: The list of event types to receive about the
// target resource. When using the `*` wildcard (equivalent to `PUT`), omitted
// fields are set to empty values and rejected if they're invalid.
func (c *SubscriptionsPatchCall) UpdateMask(updateMask string) *SubscriptionsPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// ValidateOnly sets the optional parameter "validateOnly": If set to `true`,
// validates and previews the request, but doesn't update the subscription.
func (c *SubscriptionsPatchCall) ValidateOnly(validateOnly bool) *SubscriptionsPatchCall {
	c.urlParams_.Set("validateOnly", fmt.Sprint(validateOnly))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SubscriptionsPatchCall) Fields(s ...googleapi.Field) *SubscriptionsPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SubscriptionsPatchCall) Context(ctx context.Context) *SubscriptionsPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SubscriptionsPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SubscriptionsPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.subscription)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "workspaceevents.subscriptions.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "workspaceevents.subscriptions.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SubscriptionsPatchCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "workspaceevents.subscriptions.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SubscriptionsReactivateCall struct {
	s                             *Service
	name                          string
	reactivatesubscriptionrequest *ReactivateSubscriptionRequest
	urlParams_                    gensupport.URLParams
	ctx_                          context.Context
	header_                       http.Header
}

// Reactivate: Reactivates a suspended Google Workspace subscription. This
// method resets your subscription's `State` field to `ACTIVE`. Before you use
// this method, you must fix the error that suspended the subscription. This
// method will ignore or reject any subscription that isn't currently in a
// suspended state. To learn how to use this method, see Reactivate a Google
// Workspace subscription
// (https://developers.google.com/workspace/events/guides/reactivate-subscription).
// For a subscription on a Chat target resource
// (https://developers.google.com/workspace/events/guides/events-chat), you can
// reactivate a subscription as: - A Chat app by specifying an authorization
// scope that begins with `chat.app` andgetting one-time administrator approval
// (Developer Preview (https://developers.google.com/workspace/preview)). To
// learn more, see Authorize as a Chat app with administrator approval
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-app).
// - A user by specifying an authorization scope that doesn't include `app` in
// its name. To learn more, see Authorize as a Chat user
// (https://developers.google.com/workspace/chat/authenticate-authorize-chat-user).
//
//   - name: Resource name of the subscription. Format:
//     `subscriptions/{subscription}`.
func (r *SubscriptionsService) Reactivate(name string, reactivatesubscriptionrequest *ReactivateSubscriptionRequest) *SubscriptionsReactivateCall {
	c := &SubscriptionsReactivateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.reactivatesubscriptionrequest = reactivatesubscriptionrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SubscriptionsReactivateCall) Fields(s ...googleapi.Field) *SubscriptionsReactivateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SubscriptionsReactivateCall) Context(ctx context.Context) *SubscriptionsReactivateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SubscriptionsReactivateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SubscriptionsReactivateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.reactivatesubscriptionrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:reactivate")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "workspaceevents.subscriptions.reactivate", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "workspaceevents.subscriptions.reactivate" call.
// Any non-2xx status code is an error. Response headers are in either
// *Operation.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SubscriptionsReactivateCall) Do(opts ...googleapi.CallOption) (*Operation, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Operation{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "workspaceevents.subscriptions.reactivate", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TasksCancelCall struct {
	s                 *Service
	name              string
	canceltaskrequest *CancelTaskRequest
	urlParams_        gensupport.URLParams
	ctx_              context.Context
	header_           http.Header
}

// Cancel: Cancel a task from the agent. If supported one should expect no more
// task updates for the task.
//
// - name: The resource name of the task to cancel. Format: tasks/{task_id}.
func (r *TasksService) Cancel(name string, canceltaskrequest *CancelTaskRequest) *TasksCancelCall {
	c := &TasksCancelCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.canceltaskrequest = canceltaskrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TasksCancelCall) Fields(s ...googleapi.Field) *TasksCancelCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TasksCancelCall) Context(ctx context.Context) *TasksCancelCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TasksCancelCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TasksCancelCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.canceltaskrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:cancel")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "workspaceevents.tasks.cancel", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "workspaceevents.tasks.cancel" call.
// Any non-2xx status code is an error. Response headers are in either
// *Task.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *TasksCancelCall) Do(opts ...googleapi.CallOption) (*Task, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Task{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "workspaceevents.tasks.cancel", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TasksGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Get the current state of a task from the agent.
//
// - name: The resource name of the task. Format: tasks/{task_id}.
func (r *TasksService) Get(name string) *TasksGetCall {
	c := &TasksGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// HistoryLength sets the optional parameter "historyLength": The number of
// most recent messages from the task's history to retrieve.
func (c *TasksGetCall) HistoryLength(historyLength int64) *TasksGetCall {
	c.urlParams_.Set("historyLength", fmt.Sprint(historyLength))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TasksGetCall) Fields(s ...googleapi.Field) *TasksGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *TasksGetCall) IfNoneMatch(entityTag string) *TasksGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TasksGetCall) Context(ctx context.Context) *TasksGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TasksGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TasksGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "workspaceevents.tasks.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "workspaceevents.tasks.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Task.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *TasksGetCall) Do(opts ...googleapi.CallOption) (*Task, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Task{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "workspaceevents.tasks.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TasksSubscribeCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Subscribe: TaskSubscription is a streaming call that will return a stream of
// task update events. This attaches the stream to an existing in process task.
// If the task is complete the stream will return the completed task (like
// GetTask) and close the stream.
//
//   - name: The resource name of the task to subscribe to. Format:
//     tasks/{task_id}.
func (r *TasksService) Subscribe(name string) *TasksSubscribeCall {
	c := &TasksSubscribeCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TasksSubscribeCall) Fields(s ...googleapi.Field) *TasksSubscribeCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *TasksSubscribeCall) IfNoneMatch(entityTag string) *TasksSubscribeCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TasksSubscribeCall) Context(ctx context.Context) *TasksSubscribeCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TasksSubscribeCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TasksSubscribeCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:subscribe")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "workspaceevents.tasks.subscribe", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "workspaceevents.tasks.subscribe" call.
// Any non-2xx status code is an error. Response headers are in either
// *StreamResponse.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *TasksSubscribeCall) Do(opts ...googleapi.CallOption) (*StreamResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &StreamResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "workspaceevents.tasks.subscribe", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TasksPushNotificationConfigsCreateCall struct {
	s                          *Service
	parent                     string
	taskpushnotificationconfig *TaskPushNotificationConfig
	urlParams_                 gensupport.URLParams
	ctx_                       context.Context
	header_                    http.Header
}

// Create: Set a push notification config for a task.
//
// - parent: The parent task resource for this config. Format: tasks/{task_id}.
func (r *TasksPushNotificationConfigsService) Create(parent string, taskpushnotificationconfig *TaskPushNotificationConfig) *TasksPushNotificationConfigsCreateCall {
	c := &TasksPushNotificationConfigsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.taskpushnotificationconfig = taskpushnotificationconfig
	return c
}

// ConfigId sets the optional parameter "configId": Required. The ID for the
// new config.
func (c *TasksPushNotificationConfigsCreateCall) ConfigId(configId string) *TasksPushNotificationConfigsCreateCall {
	c.urlParams_.Set("configId", configId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TasksPushNotificationConfigsCreateCall) Fields(s ...googleapi.Field) *TasksPushNotificationConfigsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TasksPushNotificationConfigsCreateCall) Context(ctx context.Context) *TasksPushNotificationConfigsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TasksPushNotificationConfigsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TasksPushNotificationConfigsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.taskpushnotificationconfig)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "workspaceevents.tasks.pushNotificationConfigs.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "workspaceevents.tasks.pushNotificationConfigs.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *TaskPushNotificationConfig.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *TasksPushNotificationConfigsCreateCall) Do(opts ...googleapi.CallOption) (*TaskPushNotificationConfig, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TaskPushNotificationConfig{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "workspaceevents.tasks.pushNotificationConfigs.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TasksPushNotificationConfigsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Delete a push notification config for a task.
//
//   - name: The resource name of the config to delete. Format:
//     tasks/{task_id}/pushNotificationConfigs/{config_id}.
func (r *TasksPushNotificationConfigsService) Delete(name string) *TasksPushNotificationConfigsDeleteCall {
	c := &TasksPushNotificationConfigsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TasksPushNotificationConfigsDeleteCall) Fields(s ...googleapi.Field) *TasksPushNotificationConfigsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TasksPushNotificationConfigsDeleteCall) Context(ctx context.Context) *TasksPushNotificationConfigsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TasksPushNotificationConfigsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TasksPushNotificationConfigsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "workspaceevents.tasks.pushNotificationConfigs.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "workspaceevents.tasks.pushNotificationConfigs.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *TasksPushNotificationConfigsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "workspaceevents.tasks.pushNotificationConfigs.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TasksPushNotificationConfigsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Get a push notification config for a task.
//
//   - name: The resource name of the config to retrieve. Format:
//     tasks/{task_id}/pushNotificationConfigs/{config_id}.
func (r *TasksPushNotificationConfigsService) Get(name string) *TasksPushNotificationConfigsGetCall {
	c := &TasksPushNotificationConfigsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TasksPushNotificationConfigsGetCall) Fields(s ...googleapi.Field) *TasksPushNotificationConfigsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *TasksPushNotificationConfigsGetCall) IfNoneMatch(entityTag string) *TasksPushNotificationConfigsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TasksPushNotificationConfigsGetCall) Context(ctx context.Context) *TasksPushNotificationConfigsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TasksPushNotificationConfigsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TasksPushNotificationConfigsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "workspaceevents.tasks.pushNotificationConfigs.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "workspaceevents.tasks.pushNotificationConfigs.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *TaskPushNotificationConfig.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *TasksPushNotificationConfigsGetCall) Do(opts ...googleapi.CallOption) (*TaskPushNotificationConfig, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TaskPushNotificationConfig{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "workspaceevents.tasks.pushNotificationConfigs.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TasksPushNotificationConfigsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Get a list of push notifications configured for a task.
//
// - parent: The parent task resource. Format: tasks/{task_id}.
func (r *TasksPushNotificationConfigsService) List(parent string) *TasksPushNotificationConfigsListCall {
	c := &TasksPushNotificationConfigsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": For AIP-158 these fields
// are present. Usually not used/needed. The maximum number of configurations
// to return. If unspecified, all configs will be returned.
func (c *TasksPushNotificationConfigsListCall) PageSize(pageSize int64) *TasksPushNotificationConfigsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token received
// from a previous ListTaskPushNotificationConfigRequest call. Provide this to
// retrieve the subsequent page. When paginating, all other parameters provided
// to `ListTaskPushNotificationConfigRequest` must match the call that provided
// the page token.
func (c *TasksPushNotificationConfigsListCall) PageToken(pageToken string) *TasksPushNotificationConfigsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TasksPushNotificationConfigsListCall) Fields(s ...googleapi.Field) *TasksPushNotificationConfigsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *TasksPushNotificationConfigsListCall) IfNoneMatch(entityTag string) *TasksPushNotificationConfigsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TasksPushNotificationConfigsListCall) Context(ctx context.Context) *TasksPushNotificationConfigsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TasksPushNotificationConfigsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TasksPushNotificationConfigsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/pushNotificationConfigs")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "workspaceevents.tasks.pushNotificationConfigs.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "workspaceevents.tasks.pushNotificationConfigs.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListTaskPushNotificationConfigResponse.ServerResponse.Header or (if a
// response was returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *TasksPushNotificationConfigsListCall) Do(opts ...googleapi.CallOption) (*ListTaskPushNotificationConfigResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListTaskPushNotificationConfigResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "workspaceevents.tasks.pushNotificationConfigs.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *TasksPushNotificationConfigsListCall) Pages(ctx context.Context, f func(*ListTaskPushNotificationConfigResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}
